﻿

namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/module-master")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ModulesMasterController : BaseController
    {
        /// <summary>
        /// The ModulesMaster services.
        /// </summary>
        private readonly IModulesMasterService modulesMastersService;
        /// <summary>
        /// The ModulesMaster services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public ModulesMasterController(IModulesMasterService modulesMasterService, IAuditLogService auditLogServices)
        {
            this.modulesMastersService = modulesMasterService;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The fetch async.
        /// </summary>
        /// <returns>
        /// The <see cref="Task"/>
        /// </returns>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ModulesMasterModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] ModulesMasterModel model)
        {
            model ??= new ModulesMasterModel();
            var modules = await this.modulesMastersService.FetchModules(model);
            return this.Success(modules);

        }

        /// <summary>
        /// The create async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> CreateAsync([FromBody] ModulesMasterModel model)
        {
            model = (ModulesMasterModel)EmptyFilter.Handler(model);
            var response = await this.modulesMastersService.AddModule(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.ModuleCharge,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $@"<b>{model.CreatedByName}</b> has added the Master Module <b> {model.ModuleName}</b>  successfully",
                LocationId = Convert.ToInt32(model.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="model"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] ModulesMasterModel model)
        {
            model = (ModulesMasterModel)EmptyFilter.Handler(model);
            var response = await this.modulesMastersService.UpdateModule(model);
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.ModuleCharge,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $@"<b>{model.CreatedByName}</b> has updated the Master Module <b> {model.ModuleName}</b>  successfully",
                LocationId = Convert.ToInt32(model.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);
            return this.Success(response);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="model"></param>
        /// <returns></returns>
        [HttpPost]
        [Route("delete-modules")]
        public async Task<ActionResult> DeleteAsync([FromBody] ModulesMasterModel model)
        {
            try
            {
                model = (ModulesMasterModel)EmptyFilter.Handler(model);
                var response = await this.modulesMastersService.DeleteModule(model);
                if (response == 0)
                {
                    return this.ServerError();
                }

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.ModuleCharge,
                    LogFrom = (short)model.LoginRoleId,
                    LogDate = DateTime.UtcNow.AddMinutes(330),
                    LogDescription = $@"<b>{model.ModifiedByName}</b> has deleted the Master Module <b> {model.ModuleName}</b>  successfully",
                    LocationId = Convert.ToInt32(model.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
                return this.Success("Module Master has been deleted Successfully.");
            }
            catch(NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("Module master can't be deleted. Please contact Administrator.");
                }
                return this.ServerError();
            }
            
        }
    }
}
